<?php
/**
 * This file is part of
 * pragmaMx - Web Content Management System.
 * Copyright by pragmaMx Developer Team - http://www.pragmamx.org
 *
 * pragmaMx is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * $Revision: 626 $
 * $Author: PragmaMx $
 * $Date: 2022-09-19 08:11:22 +0200 (Mo, 19. Sep 2022) $
 */

/**
 * pmxHeader
 *
 * @package pragmaMx
 * @author tora60
 * @copyright Copyright (c) 2008
 * @version $Id: Header.php 626 2022-09-19 06:11:22Z PragmaMx $
 * @access public
 */
class pmxHeader {

    /**
     * Ready for upgrade jQuery + UI
     * Use this for development
     * INFO: Set to true, will show all old files errors otherwise to false
     * 
     * @access private
     * @var bool $__jQueryOldErrorMessage
     * @default false
     */
    private static $__jQueryOldErrorMessage = false;

    /**
     * Require to get an uniqId
     * 
     * @private
     * @var int $_UNIQID
     * @default 0
     */
    private static $_UNIQID = 0;

    /**
     * @private
     * @var array $__adds
     * @var array $__adds["body"]
     * @var array $__adds["body_code"]
     * @var array $__adds["more"]
     * @var array $__adds["script"]
     * @var array $__adds["script_code"]
     * @var array $__adds["script_code_docu"]
     * @var array $__adds["script_code_func"]
     * @var array $__adds["style"]
     * @var array $__adds["style_code"]
     */
    private static $__adds = [
        'body' => [],
        'body_code' => [],
        'more' => [],
        'script' => [],
        'script_code' => [],
        'script_code_docu' => [],
        'script_code_func' => [],
        'style' => [],
        'style_code' => [],
    ];

    /**
     * Eine Liste aller urls, die eingefügt werden sollen.
     * Verhindern, dass diese doppelt eingefügt werden.
     * 
     * @private
     * @var array $__urls
     * @default array()
     */
    private static $__urls = array();

    /**
     * Meta Informationen
     * 
     * @private
     * @var array $__meta
     */
    private static $__meta = array(
        'title' => "",
        'description' => "",
        'robots' => "index,follow",
        'canonical' => "",
        'revisit' => "8",
        'copyright' => "",
        'resource-type' => "document",
        'distribution' => "global",
        'rating' => "general",
        'author' => "",
        'alternate' => "",
		'viewport' =>"",
    );

    /**
     * @private
     * @var array $__keywords
     * @default array()
     */
    private static $__keywords = array();

    /**
     * jQuery Details
     * 
     * @private
     * @var array $__jquery
     */
    private static $__jquery = array(
        'required' => false,
        'loaded' => false,
        'path' => 'includes/javascript/jquery/',
        'main' => 'jquery.js',
        'main_min' => 'jquery.min.js',
        'files' => array(),
        'ui_required' => false,
        'ui_loaded' => false,
        'ui_main' => 'ui/jquery.ui.core.js', // Datei gibt es nicht, variable wird aber gebraucht
        'ui_main_min' => 'ui/jquery-ui-pmx-core.min.js', // jQuerUI 1.10.2
        'ui_theme' => 'default',
        'ui_theme_path' => 'layout/jquery/ui/',
        'ui_theme_default' => 'default' // jQueryUI >v1.13 hat "base" und kein "default"
    );
	
    /**
     * @private
     * @var $__status
     */
	private static $__status = NULL;

    /**
     * pmxHeader::__construct()
     * 
     * @private - verhindert die direkte Erzeugung des Objektes
     */
    private function __construct() {}

    /**
     * Returns content of the output
     *
     * @returns array
     */
    public static function get()
    {
        $buf = self::_fetch();
        $buf = trim(implode("\n", $buf));
        // unnötige spaces entfernen
        $buf = trim(preg_replace('#\s*\n\s+#', "\n", $buf));
		//mxDebugFuncVars($buf);
        return $buf;
    }

    /**
     * Outputs content of the buffer and delete the buffer
     *
     * @returns string
     * @static
     */
    public static function show()
    {
        // Puffer ausgeben
        echo '<!-- start~header~' . MX_TIME . '~pmx -->', self::get(), '<!-- end~header~' . MX_TIME . '~pmx -->';
    }

    /**
     * pmxHeader::move()
     *
     * @param mixed $showed
     * @return
     */
    public static function move($showed)
    {
        $pattern = '#<!-- start~header~' . MX_TIME . '~pmx -->.*?<!-- end~header~' . MX_TIME . '~pmx -->#s';
        if (preg_match($pattern, $showed, $matches)) {
            $showed = str_replace($matches[0], self::get(), $showed);
        }
        return $showed;
    }

    /**
     * self::_fetch()
     * Einstellungen prüfen und komplettieren
     *
     * @param mixed $defaults
     * @return
     * @private
     */
    private static function _fetch($defaults = false)
    {
        $more = self::_get_more(); // als erstes abfragen, weil in my_header noch Zuweisungen stehen könnten
        $export['style'] = "\n" . '<link rel="stylesheet" type="text/css" href="' . MX_THEME_DIR . '/style/style.css" />' . "\n";
        $export['jquery'] = self::_get_jquery();
        $export['style'] .= self::_get_style();
        $export['style_code'] = self::_get_style_code();
        $export['script_code'] = self::_get_script_code();
        $export['script'] = self::_get_script();
        $export["script_code_func"] = self::_getScriptCodeFunc();
        $export["script_code_docu"] = self::_getScriptCodeDocu();
        $export['more'] = $more;
        return $export;
    }

    /**
     * pmxHeader::add()
     * Adds code to output buffer
     *
     * @param string|array $code code for output
     * @param bool $extract
     * @return
     */
    public static function add($code, $extract = false)
    {
        if (is_array($code)) {
            foreach ($code as $item) {
                if ($extract) {
                    $item = self::_extract($item);
                }
                self::$__adds['more'][] = $item;
            }
        } else {
            if ($extract) {
                $code = self::_extract($code);
            }
            self::$__adds['more'][] = $code;
        }
    }

    /**
     * Das ist die neue Funktion die add_script(), add_body_script(), add_style() verreint
     * 
     * @param string $type
     * @param mixed $file
     * @param string [$media=""]
     * @param string [$if_for_ie_version=""]
     */
    private static function _addFile($type, $file, $media = "", $if_for_ie_version = "")
    {
        if (!in_array($file, self::$__urls)) {
            self::$__urls[] = $file;
            // Systempfade am Anfang entfernen
			$file1=$file;
            $file = trim(self::_cleanpath($file));
            if ($file=="") {
				trigger_error("_addFile()0 - Filename empty - " . $file ." Type: ".$type, E_USER_NOTICE);
				return false;
			}

            //mxDebugFuncvars($file1,$file);

            switch($type) {
                case 'script': // add_script()
                case 'body': // add_body_script()
                    $file = self::_getminfilename($file);
                    
                    if (self::_isJqueryMainFile($file, false)) {
                        self::$__jquery['required'] = true;
                    } elseif (self::_isJqueryMainFile($file, true)) {
                        self::$__jquery['required'] = true;
                        self::$__jquery['ui_required'] = true;
                    } elseif (file_exists($file) || self::_isFileHttp($file)) {
                        $file = '<script type="text/javascript" src="' . $file . '"></script>';
                        if ($if_for_ie_version) {
                            $file = "<!--[if " . $if_for_ie_version . "]>\n" . $file . "\n<![endif]-->";
                        }
                        self::$__adds[$type][] = $file;
                    } else {
                        trigger_error("_addFile() script - File not found - " . $file1, E_USER_NOTICE);
                    }
                    break;

                case 'style': // add_style()
					//mxDebugFuncVars($file);

                    if (file_exists($file1) ) {
                        $media = $media ? ' media="' . $media . '"' : "";
                        $file = '<link rel="stylesheet" type="text/css" href="' . $file . '"' . $media . ' />';
                        if ($if_for_ie_version) {
                            $file = "<!--[if " . $if_for_ie_version . "]>\n" . $file . "\n<![endif]-->";
                        }
                        self::$__adds[$type][] = $file;
                    } else {
                        //trigger_error("_addFile() style - File not found - " . $file . " - ".$file1, E_USER_NOTICE);
                    }
                    break;

                default:
                    trigger_error("_addFile()3 - Type not found - " . $type, E_USER_NOTICE);
                    break;
            }
        } // else $file schon drin
    }

    /**
     * pmxHeader::add_style()
     *
     * @param string $file
     * @param string [$media=""]
     * @param string [$if_for_ie_version=""]
     */
    public static function add_style($file, $media = '', $if_for_ie_version = '')
    {
        self::_addFile("style", $file, $media, $if_for_ie_version);
    }

    /**
     * pmxHeader::add_style_code()
     *
     * @param mixed $code
     * @return
     */
    public static function add_style_code($code)
    {
        self::$__adds['style_code'][] = preg_replace(array('#</?style[^>]*>#i', '#[[:cntrl:]]#', '#\s+#'), array('', '', ' '), $code);
    }

    /**
     * pmxHeader::add_script()
     *
     * @param string $file
     * @param string [$if_for_ie_version=""]
     * @return
     */
    public static function add_script($file, $if_for_ie_version = '')
    {
        self::_addFile("script", $file, "", $if_for_ie_version);
    }

    /**
     * pmxHeader::add_script_code()
     *
     * @param mixed $code
     * @return
     */
    public static function add_script_code($code)
    {
        self::$__adds['script_code'][] = preg_replace('#</?script[^>]*>#i', '', $code);
    }
 
    /**
     * pmxHeader::add_body()
     *
     * @param mixed $code
     * @param [$extract=false]
     * @return
     */
    public static function add_body($code, $extract = false)
    {
        if (is_array($code)) {
            foreach ($code as $item) {
                if ($extract) {
                    $item = self::_extract($item);
                }
                self::$__adds['body'][] = $item;
            }
        } else {
            if ($extract) {
                $code = self::_extract($code);
            }
            self::$__adds['body'][] = $code;
        }
    }   

	/**
     * pmxHeader::add_script_body()
     *
     * @param mixed $code
     * @return
     */
    public static function add_body_script_code($code)
    {
        self::$__adds['body_code'][] = preg_replace('#</?script[^>]*>#i', '', $code);
    }
	
    /**
     * pmxHeader::add_body_script()
     *
     * @param mixed $src
     * @param string [$if_for_ie_version=""]
     */
    public static function add_body_script($src, $if_for_ie_version = '')
    {
        self::_addFile("body", $src, "", $if_for_ie_version);
    }

    /**
     * pmxHeader::add_jquery()
     *
     * @return
     */
    public static function add_jquery()
    {
        // immer erforderlich
        self::$__jquery['required'] = true;

        if (func_num_args() > 0) {
            $args = func_get_args();
            foreach ($args as $src) {
				
                if (!in_array($src, self::$__urls)) {
                    self::$__urls[] = $src;
                    // "ui" - ready for jQueryUI 1.13.1
                    if ($src === "ui" || self::_isJqueryMainFile($src, true)) {
                        self::$__jquery['ui_required'] = true;
                    }
                    // alle jQueryUI plugins
                    elseif (strpos(basename($src), 'jquery.ui.') === 0) {
                        self::$__jquery['ui_required'] = true;
                        self::$__jquery['files'][] = $src;
                    }
                    // alle jQuery plugins
                    else {
                        self::$__jquery['files'][] = $src;
                    }
                } // else $src schon drin
				
            }
			
        }
    }

    /**
     * pmxHeader::add_jquery_ui()
     *
     * @param string [$theme] - Optional, set to default theme
     */
	public static function add_jquery_ui($theme) {
        // alles unnötige aus dem Namen entfernen
        $theme = preg_replace('/[^a-z0-9\-]/i', '', $theme);
        $theme = $theme ? $theme : self::$__jquery["ui_theme_default"];
		pmxBase::set_system("juitheme", $theme);
	}

    /**
     * pmxHeader::add_lightbox()
     *
     * @param mixed $options
     */
    public static function add_lightbox($options = false)
    {
        switch (true) {
            case is_array($options):
            case is_object($options):
                $options = '?' . http_build_query($options, 'x', '&amp;');
                break;
            case $options:
                // nur reset
                $options = '?t=' . intval(MX_TIME);
                break;
            default:
                $options = '';
        }

        self::add_style(PMX_LAYOUT_PATH . 'jquery/css/prettyPhoto.css');
        self::add_jquery('jquery.prettyPhoto.js', 'lightbox.js.php' . $options);
    }

    /**
     * pmxHeader::add_prettyphoto()
     * clone von add_lightbox()
     *
     * @param mixed [$options=false]
     */
    public static function add_prettyphoto($options = false)
    {
        self::add_lightbox($options);
    }

    /**
     * pmxHeader::add_tabs()
     *
     * @param bool [$cookie=true]
     */
    public static function add_tabs($cookie = true)
    {
        if ($cookie === true) {
            self::add_jquery('jquery.cookie.js');
        }
        self::add_jquery('ui/jquery.ui.tabs.noui.js');
    }

    /**
     * self::_get_style()
     *
     * @return string
     * @private
     */
    private static function _get_style()
    {
        return implode("\n", array_unique(self::$__adds['style'])) . "\n";
    }

    /**
     * self::_get_style_code()
     *
     * @return string
     */
    private static function _get_style_code()
    {
        if (array_key_exists('style_code', self::$__adds)) {
            return self::_preparelines(self::$__adds['style_code'], 'style') . "\n";
        }
        return "";
    }

    /**
     * self::_get_script()
     *
     * @return string
     * @private
     */
    private static function _get_script()
    {
        if (array_key_exists('script', self::$__adds)) {
            return implode("\n", array_unique(self::$__adds['script'])) . "\n";
        }
        return "";
    }

    /**
     * self::_get_script_code()
     *
     * @return string
     * @private
     */
    private static function _get_script_code()
    {
        if (array_key_exists('script_code', self::$__adds)) {
            return self::_preparelines(self::$__adds['script_code'], 'script') . "\n";
        }
        return "";
    }

    /**
     * self::_get_jquery()
     *
     * @return string
     * @private
     */
    private static function _get_jquery()
    {

        $_src = [];
        // trim ?
        
        // aufräumen
        self::$__jquery['files'] = array_unique(self::$__jquery['files']);

        // alle Dateien prüfen
		$ol=self::$__jquery['files'];
        foreach (self::$__jquery['files'] as $key => $value) {
            if (self::_isJqueryMainFile($value, true)) {
                continue;
            }
            $_min = self::_getminfilename(self::$__jquery['path'] . $value);
			$_min = str_replace("//","/",$_min);
            if (file_exists($_min)) {
                $_src[] = '<script type="text/javascript" src="' . $_min . '"></script>';
            }
			//mxDebugFuncVars($value,$_min,self::$__jquery['path']);
        }
		

        // zuerst jQueryUI
        if (self::$__jquery['ui_required'] && !self::$__jquery['ui_loaded']) {
            self::_setUITheme();
            array_unshift(
                $_src,
                '<script type="text/javascript" src="' . self::$__jquery['path'] . self::$__jquery['main_min'] . '"></script>',
                '<script type="text/javascript" src="' . self::$__jquery['path'] . self::$__jquery['ui_main_min'] . '"></script>'
            );
            self::$__jquery['loaded'] == true;
            self::$__jquery['ui_loaded'] == true;
        }

        // oder jQuery
        elseif (self::$__jquery['required'] && !self::$__jquery['loaded']) {
            array_unshift(
                $_src,
                '<script type="text/javascript" src="' . self::$__jquery['path'] . self::$__jquery['main_min'] . '"></script>'
            );
            self::$__jquery['loaded'] == true;
        }
		//mxDebugFuncVars($_src);
        return implode("\n", $_src) . "\n";
    }

    /**
     * self::_setUITheme()
     * 
     * TODO: flexibler machen, so dass es über das Theme gesteuert werden könnte 
     */
    private static function _setUITheme()
    {
        $path = trim(self::$__jquery['ui_theme_path'], ' /') . '/' . pmxBase::juitheme();
        if (file_exists($path)) {
            self::$__jquery['ui_theme'] = pmxBase::juitheme();
        } else {
            // Fallback für falsch eingestelltes UI-Theme
            self::$__jquery['ui_theme'] = self::$__jquery['ui_theme_default'];
            $path = trim(self::$__jquery['ui_theme_path'], ' /') . '/' . self::$__jquery['ui_theme'];
        }

        // Stylesheet für jQuery-UI anfügen
        self::add_style(self::_getminfilename($path . '/jquery-ui.css', 'css'));
        // optionales Stylesheet anfügen
        self::add_style(self::_getminfilename(PMX_LAYOUT_PATH. 'jquery/jquery-ui.custom.css', 'css'));
    }

    /**
     * self::_get_more()
     *
     * @return string
     * @private
     */
    private static function _get_more()
    {
        $more = array_unique(self::$__adds['more']);
        foreach ($more as $key => $value) {
            $more[$key] = preg_replace('#\s*\n\s+#', "\n", $value);
        }
        return implode("\n", $more) . "\n";
    }

    /**
     * pmxHeader::get_body()
     * setzt zusätzliche JScrips ans ende des HTML-Body's
     * 
     * @param string $body_content
     * @return string
     */
    public static function get_body($body_content)
    {
		$body_js = "\n";
        // die Sammlung zurückgeben...
        $body_js .= implode("\n", array_unique(self::$__adds['body'])) . "\n";
        $body = array_unique(self::$__adds['body_code']);
        $body_js .= self::_preparelines($body, 'script') . "\n";
    
		return $body_content . "\n" . $body_js;
    }

    /**
     * pmxHeader::get_user()
     * nur noch zur Abwärtskompatibilität
     * my_header.php gibt es nicht mehr, wurde ersetzt durch hook 'prepare.header'
     *
     * @return string
     * @deprecated
     */
    public static function get_user()
    {
        trigger_error("@deprecated pmxHeader::get_user()", E_USER_NOTICE);
        return '';
    }

    /**
     * pmxHeader::get_meta()
     * gibt die Meta-Informationen zurück
     *
     * @return string
     */
    public static function get_meta($tag, $default = '')
    {
        // SEO Konfiguration laden
        $seo = load_class('Config', 'pmx.seo');

        $temp = "";
        switch ($tag) {
            case "keywords":
                $limit = 25;
                $keywords = $seo->metakeywords;
				if (!is_array($keywords)) $keywords=array();
                $keywords = array_merge(self::$__keywords, $keywords);
                $keywords[] = PMX_VERSION;
                $keywords = array_chunk(array_unique($keywords), $limit);
                $temp = self::_escape(implode(', ', $keywords[0]));
				//mxDebugFuncVars($keywords,$temp);
                break;

            case "title":
                global $pagetitle;
                $temp = $GLOBALS['sitename'];
                switch (true) {
                    case defined('MX_HOME_FILE'):
                        // wenn Startseite, nur den Seitenname als Seitentitel anzeigen.
                        break;
                    case trim(self::$__meta['title']):
                        $temp .= ' - ' . self::_escape(self::$__meta['title']);
                        $pagetitle = self::_escape(self::$__meta['title']); // wird in manchen Modulen weiter verwendet
                        break;
                    case trim($pagetitle):
                        $temp .= ' - ' . $pagetitle;
                        break;
                }
                // sicherstellen, dass der Seitentitel keine Tags enthält und Sonderzeichen nicht zerstückelt werden
                $temp = trim(str_replace('&nbsp;', ' ', self::_escape($temp)), '- ');
                break;

            case "author":
                if (!trim(self::$__meta['author'])) {
                    self::$__meta['author'] = $GLOBALS['sitename'];
                }
                $temp = self::_escape(self::$__meta['author']);
                break;

            case "rating":
                $oks = array('general', 'mature', 'restricted', '14 years', 'safe for kids');
                $rate = trim(self::$__meta['rating']);
                if (!($rate && in_array($rate, $oks))) {
                    self::$__meta['rating'] = "general";
                }
                $temp = self::$__meta['rating'];
                break;

            case "description":
                if (!trim(self::$__meta['description'])) {
                    self::$__meta['description'] = $GLOBALS['slogan'];
                }
                $temp = self::_escape(self::$__meta['description']);
                break;

            case "robots":
                $temp = trim(strtolower(self::$__meta['robots']));
                if (!($temp && preg_match('#^(noindex|index|nofollow|follow|archive|noarchive)\s*,\s*(noindex|index|nofollow|follow|archive|noarchive)$#', $temp))) {
                    //$temp = "index, follow";
                }
                break;

            case "canonical":
                if (trim(self::$__meta['canonical'])) {
                    $temp = self::_escape(self::$__meta['canonical']);
                }
                break;

            case "revisit":
                if (intval(self::$__meta['revisit']) == 0) {
                    self::$__meta['revisit'] = 10;
                }
                $temp = intval(self::$__meta['revisit']) . " days";
                break;

            case "copyright":
                if (!trim(self::$__meta['copyright'])) {
                    self::$__meta['copyright'] = date('Y') . ' by ' . $GLOBALS['sitename'];
                }
                $temp = self::_escape(self::$__meta['copyright']);
                break;

            case "alternate":
                // self::$__meta['alternate'] = 'meta name="revisit-after" content="1 month"';
                $temp = trim(stripslashes(strip_tags(self::$__meta['alternate'])));
                if ($temp) {
                    $temp = preg_split('#\s*[,]\s*#', $temp);
                    $temp = "<" . implode(" />\n<", $temp) . " />\n";
                }
                break;

            case "viewport":
                if (trim(self::$__meta['viewport'])) {
                    $temp = trim(self::$__meta['viewport']);
                }
                break;

            default:
                if (array_key_exists($tag, self::$__meta)) {
                    $temp = self::$__meta[$tag];
                }
                break;
        }
        return $temp;
    }

    /**
     * pmxHeader::set_meta()
     * set meta description
     *
     * @param string $tag
     * @param string $content
     */
    public static function set_meta($tag, $content)
    {
        self::$__meta[$tag] = strip_tags($content);
    }
	
	/**
     * pmxHeader::status()
     * set meta description
     *
     * @param $code
     * @return
     */
    public static function Status($code = NULL)
    {
        if ($code === NULL) {
            if (self::$__status == NULL) {
                self::$__status = isset($GLOBALS['http_response_code']) ? $GLOBALS['http_response_code'] : http_response_code();
            }
        } else { 
            http_response_code(intval($code));
            self::$__status = intval($code);
        }
        
        $protocol = isset($_SERVER['SERVER_PROTOCOL']) ? $_SERVER['SERVER_PROTOCOL'] : 'HTTP/1.0';
        
        switch (self::$__status) {
            case 100: $text = 'Continue'; break;
            case 101: $text = 'Switching Protocols'; break;
            case 200: $text = 'OK'; break;
            case 201: $text = 'Created'; break;
            case 202: $text = 'Accepted'; break;
            case 203: $text = 'Non-Authoritative Information'; break;
            case 204: $text = 'No Content'; break;
            case 205: $text = 'Reset Content'; break;
            case 206: $text = 'Partial Content'; break;
            case 300: $text = 'Multiple Choices'; break;
            case 301: $text = 'Moved Permanently'; break;
            case 302: $text = 'Moved Temporarily'; break;
            case 303: $text = 'See Other'; break;
            case 304: $text = 'Not Modified'; break;
            case 305: $text = 'Use Proxy'; break;
            case 400: $text = 'Bad Request'; break;
            case 401: $text = 'Unauthorized'; break;
            case 402: $text = 'Payment Required'; break;
            case 403: $text = 'Forbidden'; break;
            case 404: $text = 'Not Found'; break;
            case 405: $text = 'Method Not Allowed'; break;
            case 406: $text = 'Not Acceptable'; break;
            case 407: $text = 'Proxy Authentication Required'; break;
            case 408: $text = 'Request Time-out'; break;
            case 409: $text = 'Conflict'; break;
            case 410: $text = 'Gone'; break;
            case 411: $text = 'Length Required'; break;
            case 412: $text = 'Precondition Failed'; break;
            case 413: $text = 'Request Entity Too Large'; break;
            case 414: $text = 'Request-URI Too Large'; break;
            case 415: $text = 'Unsupported Media Type'; break;
            case 423: $text = 'Locked'; break;
            case 500: $text = 'Internal Server Error'; break;
            case 501: $text = 'Not Implemented'; break;
            case 502: $text = 'Bad Gateway'; break;
            case 503: $text = 'Service Unavailable'; break;
            case 504: $text = 'Gateway Time-out'; break;
            case 505: $text = 'HTTP Version not supported'; break;
            default:
                exit('Unknown http status code "' . htmlentities($code) . '"');
            break;
        }
        
        http_response_code(self::$__status);
        $code = $protocol . ' ' . self::$__status . ' ' . $text;
        $GLOBALS['http_response_code'] = self::$__status;

        return $code;        
    }

    /**
     * pmxHeader::set_keywords()
     * set and replace existing keywords
     *
     * @param mixed $keywords
     * @return
     */
    public static function set_keywords($keywords)
    {
        if (!is_array($keywords)) {
            $keywords = preg_split('#\s*[,;]\s*#', $keywords);
        }
        self::$__keywords = $keywords;
    }

    /**
     * pmxHeader::add_keywords()
     * add keywords to existing keywords
     *
     * @param mixed $keywords
     * @return
     */
    public static function add_keywords($keywords)
    {	
        if (!is_array($keywords) and $keywords!=NULL ) {
		// mxDebugFuncVars(self::$__keywords, $keywords);
            $keywords = preg_split('#\s*[,;]\s*#', $keywords);
			self::$__keywords = array_merge(self::$__keywords, $keywords);
        }
	    
    }
    
    /**
     * pmxHeader::get_keywords()
     * get keywords 
     *
     * @param $limit
     * @param $with_seo_tags
     * @return array ($keywords)
     */
    public static function get_keywords($limit = 25, $with_seo_tags = true)
    {	
	    $seo = load_class('Config', 'pmx.seo');
        $keywords = $with_seo_tags ? $seo->metakeywords : array();
        $keywords = array_merge(self::$__keywords, $keywords);
        $keywords = array_chunk(array_unique($keywords), $limit);
		//mxDebugFuncVars($limit,$with_seo_tags,$keywords);
	    return (count($keywords)?$keywords[0]:false);
    }

    /**
     * pmxHeader::set_title()
     * set meta pagetitle
     *
     * @param string $pagetitle
     */
    public static function set_title($pagetitle)
    {
        if ($pagetitle) {
            // Titel nur setzen, wenn auch wirklich angegeben, nicht dass fehlerhafte Module da Mist bauen
            self::set_meta('title', $pagetitle, true);
        }
    }

    /**
     * pmxHeader::set_description()
     * set meta description
     *
     * @deprecated Funktion wird seit pragmaMx 2.0 nicht mehr verwendet.
     * @param string $description
     */
    public static function set_description($description)
    {
        self::set_meta('description', $description, true);
        trigger_error('@deprecated pmxHeader::set_description()', E_USER_DEPRECATED);
    }

    /**
     * Get an uniqId
     * 
     * @param string [$prefix="_"]
     * @return string
     */
    public static function getUniqId($prefix = "_") {
        return $prefix . self::$_UNIQID++;
    }

    /**
     * pmxHeader::add_script_code_func()
     * Add JS code inside of "$(function() { HERE })"
     * INFO: Add no double code
     * 
     * @require jQuery
     * @since pmx v2.8.1
     * @param string code
     */
    public static function add_script_code_func($code)
    {
        if (!in_array($code, self::$__adds["script_code_func"])) {
            self::$__jquery['required'] = true;
            self::$__adds["script_code_func"][] = $code;
        }
    }

    /**
     * pmxHeader::add_script_code_docu()
     * Add JS code inside of "$(document).ready(function() { HERE })"
     * INFO: Add no double code
     * 
     * @require jQuery
     * @since pmx v2.8.1
     * @param string code
     */
    public static function add_script_code_docu($code)
    {
        if (!in_array($code, self::$__adds["script_code_docu"])) {
            self::$__jquery['required'] = true;
            self::$__adds["script_code_docu"][] = $code;
        }
    }

    // ###################
    // ##### PRIVATE #####
    // ###################

    /**
     * pmxHeader::_getScriptCodeFunc()
     * 
     * @private
     * @since pmx v2.8.1
     * @return string
     */
    private static function _getScriptCodeFunc()
    {
        return self::_preparelines(self::$__adds["script_code_func"], "scriptFunc") . "\n";
    }

    /**
     * pmxHeader::_getScriptCodeDocu()
     * 
     * @private
     * @since pmx v2.8.1
     * @return string
     */
    private static function _getScriptCodeDocu()
    {
        return self::_preparelines(self::$__adds["script_code_docu"], "scriptDocu") . "\n";
    }

    /**
     * pmxHeader::_preparelines()
     *
     * @param array $lines
     * @param string $type
     * @return string
     * @private
     */
    private static function _preparelines($lines, $type)
    {
        $lines = trim(implode("\n", array_unique($lines)));
        if (!$lines) {
            return '';
        }

        // Kommentare entfernen
        $lines = str_replace(array('<![CDATA[', ']]>'), '', $lines);
        $lines = preg_replace(array('#/\*[^\*]*\*/#sU', '#\<\!--.+--\>#siU', '#\n\s*//[^\n]*#'), '', $lines);

        $_cdata = "\n /* <![CDATA[ */\n" . $lines . "\n /* ]]> */\n";
        $out = '';
        switch($type) {
            case 'script':
                $out = '<script type="text/javascript">' . $_cdata . '</script>';
                break;
            case 'style':
                $out = '<style type="text/css">' . $_cdata . '</style>';
                break;
            case "scriptDocu":
                $out = '<script type="text/javascript">$(document).ready(function() {' . $lines . "});</script>";
                break;
            case "scriptFunc":
                $out = '<script type="text/javascript">$(function() {' . $lines . "});</script>";
                break;
            default:
                trigger_error("_preparelines(), tpye not exists: " . $type, E_USER_NOTICE);
                break;
        }
        return $out;
    }

    /**
     * pmxHeader::_cleanpath()
     * schneidet Sytempfade am Anfang eines Pfades oder URL ab
     *
     * @param mixed $path
     * @return
     * @private
     */
    private static function _cleanpath($path)
    {
        // TODO: kann man das nicht kombinieren mit trim() ?
        $path = mx_strip_sysdirs($path);
        return str_replace(DS, '/', $path);
    }

    /**
     * pmxHeader::_extract()
     * filtert bestimmte Bestandteile aus gemischten Headerdaten
     *
     * @param string $code
     * @return string
     * @private
     */
    private static function _extract($code)
    {
        // conditional comments ignorieren <!--[if lte IE 6]><![endif]-->
        preg_match_all('#<\!--\s*\[if[^\]]+\]>.+<\!\[endif\]\s*-->#siU', $code, $matches);
        $condition = array();
        if ($matches) {
            if (!self::_is_ie()) {
                $code = str_replace($matches[0], '', $code);
            } else {
                foreach ($matches[0] as $key => $src) {
                    $condition['~!~' . $key . '~!~'] = $src;
                }
            }
            if ($condition) {
                $code = str_replace(array_values($condition), array_keys($condition), $code);
            }
        }

        // Scriptdateien rausfiltern und an Scriptarray anfügen
        preg_match_all('#<script[^>]+src\s*=\s*["\']([^"\']+)["\'][^>]*>\s*</script>#siU', $code, $matches);
        if ($matches) {
            $code = str_replace($matches[0], '', $code);
            foreach ($matches[1] as $src) {
                self::add_script($src);
            }
        }

        // Script- und Style-Codedateien rausfiltern und an entspr. Array anfügen
        preg_match_all('#<(script|style)[^>]*>(.+)</\1>#siU', $code, $matches);
        if ($matches) {
            $code = str_replace($matches[0], '', $code);
            foreach ($matches[2] as $key => $src) {
                if ($matches[1][$key] == 'script') {
                    self::add_script_code($src);
                } else {
                    self::add_style_code($src);
                }
            }
        }

        // Stylesheets rausfiltern und an Sheetarray anfügen
        preg_match_all('#<link[^>]+href\s*=\s*["\']([^"\']+)["\'][^>]*/?>#siU', $code, $matches);
        if ($matches) {
            foreach ($matches[1] as $key => $src) {
                if (strpos($matches[0][$key], 'stylesheet') !== false) {
                    self::add_style($src);
                    $code = str_replace($matches[0][$key], '', $code);
                }
            }
        }

        // conditional comments wieder zurückschreiben
        if ($condition && $code) {
            $code = str_replace(array_keys($condition), array_values($condition), $code);
        }

        if ($code) {
            // falls xhtml-Endung fehlt, diese anfügen
            $code = preg_replace('#<((?:img|input|hr|br|link|meta|base)(?:[^>]*[^/])?)>#i', '<$1 />', $code);
        }

        // raus damit
        return $code;
    }

    /**
     * pmxHeader::_is_ie()
     * guggen, was für ein Browser am werkeln ist
     *
     * @return bool
     * @private
     */
    private static function _is_ie()
    {
        $browser = load_class('Browser');
        return $browser->is_msie();
    }

    /**
     * Nicht in Gebrauch
     * 
     * @param string $src
     * @copyright pmx v2.8.1
     * @return void
     */
    private static function _clearstatcache($src)
    {
        // zuviele Anfragen dieser Dateien
        if (in_array($src, [
            "includes/javascript/mx_checklist.js", // AdminForm()
            "includes/filemanager/elfinder/manager/js/i18n/elfinder.da.js",
            "includes/filemanager/elfinder/manager/js/i18n/elfinder.de.js",
            "includes/filemanager/elfinder/manager/js/i18n/elfinder.es.js",
            "includes/filemanager/elfinder/manager/js/i18n/elfinder.fr.js",
            "includes/filemanager/elfinder/manager/js/i18n/elfinder.tr.js",
            "includes/wysiwyg/ckeditor/editor/ckeditor.js", // AdminForm()
            //"themes/**/js/jquery.theme.js"
        ]) || preg_grep("/jquery.theme./i", [$src])) {
            clearstatcache(true, $src);
        }
    }

    /**
     * self::_getminfilename()
     *
     * @param string $src
     * @param string [$ending="js"]
     * @return string $src
     * @private
     */
    private static function _getminfilename($src, $ending = 'js')
    {
        $ol=$src;
        $src = str_replace("\\", "/", $src);
        $src1 = str_replace("\\", "/", PMX_REAL_BASE_DIR);
        $src= str_replace($src1,"",$src);
		$src = str_replace("//", "/", $src);
        if (substr($src,0,1)=="/") $src=substr($src,1);
        
        if (!self::_isFileHttp($src)) {
            $_tmp = $src;
            self::_checkJqueryOldFiles($src, '_getminfilename()');
			
            // ausgelagert
            // self::_clearstatcache($src);
			
            // im erweiterten Debugmodus die Originale laden
            if (pmxDebug::is_mode('enhanced') && is_file($src) && file_exists($src)) {
                return $src;
            }
           
            // Endung wie gefordert?
            $tmp = explode('.', $src);
            if (array_pop($tmp) !== $ending && file_exists($src)) {
                return $src;
            }
            // bereits minimierte Version?
            $last = array_pop($tmp);
            if ($last === 'min' && file_exists($src)) {
                return $src;
            }

            // Datei vorhanden ? 
            if (file_exists($src)) {
                return $src;
            }

            // Array ergänzen + min
            array_push($tmp, $last, 'min', $ending);
            // Dateiname aus Array wieder herstellen
            $tmp = implode('.', $tmp);
            // wenn minimierte Version vorhanden, diese verwenden
            if (is_file($tmp) && file_exists($tmp)) {
                return $tmp;
            }
			
            if (self::_isExistsHashSrc($_tmp)) {
                // returns with hash
                return $_tmp;
            }

			$src1 = $_SERVER['DOCUMENT_ROOT'] . "/" . $src;
			//$src =  PMX_BASE_PATH . $src;
            // original $src
            if (file_exists($src1)) {
                return $src1;
            }

            trigger_error("_getminfilename(): File not found: " . $src, E_USER_NOTICE);
        }
        return $src;
    }

    /**
     * self::_isExistsHashSrc()
     * 
     * Check is $src with HASH
     * Like "index.php?t=1&a=2" or "index.php#top"
     * 
     * @copyright pmx v2.8.1
     * @param string $src
     * @return bool - Return true exists Hash and file
     */
    private static function _isExistsHashSrc($src)
    {
        $splitHashSrc = preg_split('/\?|\#/i', $src);
        // $splitHashSrc[1] = HASH
        if (file_exists($splitHashSrc[0])) {
            return true;
        }
        return false;
    }

    /**
     * pmxHeader::_escape()
     *
     * @param string $text
     * @return
     * @private
     */
    private static function _escape($text)
    {
        if ($text) {
            $text = htmlspecialchars(strip_tags($text), ENT_QUOTES, 'utf-8', false);
        }
        return $text;
    }

    /**
     * self::_isJqueryMainFile()
     * Check is jQuery main file
     * 
     * @since pmx 2.8.1
     * @param string $file - The file to check
     * @param bool $isUI - Check jQueryUI, then set to true
     * @return bool - Return true is a jQuery main file like "jquery.min.js"
     */
    private static function _isJqueryMainFile($file, $isUI)
    {
        $file = str_replace(self::$__jquery['path'], "", $file);
        if ($isUI === true) {
            // file
            if ($file === self::$__jquery['ui_main'] || $file === self::$__jquery['ui_main_min']) {
                return true;
            }
            return false;
        }
        // file
        if ($file === self::$__jquery['main'] || $file === self::$__jquery['main_min']) {
            return true;
        }
        return false;
    }

    /**
     * self::_isFileHttp()
     * Check file start with "http"
     * 
     * @param string $file
     */
    private static function _isFileHttp($file) {
        return strpos($file, "http") === 0;
    }

    /**
     * self::_checkJqueryOldFiles()
     * 
     * @param string $file
     * @param string $errorMessage - to better find like "add_jquery()"
     * @require self::$__jQueryOldErrorMessage
     */
    private static function _checkJqueryOldFiles($file, $errorMessage) {
        $_jQuery = [
            "files" => [
                // All diese Dateien sind in jQuery 3.6.0 mit enthalten
                 "jquery.autoresize.js",
                 "jquery.autoresize.min.js",
                 "jquery.migrate.js",
                 "jquery.migrate.min.js",
                 "jquery.timepicker.js",
                 "jquery.timepicker.min.js",
            ],
            // TODO: alle alten Plugins rausschmeissen/ersetzen/eigene schreiben
            "plugins" => [
                "jquery.tinywatermark.js",
                "jquery.tinywatermark.min.js",
                "jquery.treeview.js",
                "jquery.treeview.min.js",
                "jscroller.js", // hier schreiben wir ein eigenes
                "jscroller.min.js", // ersetzen mit jQueryUI resizable()
                "lightbox.js.php",
                "superfish.js",
                "superfish.min.js",
                "ajaxupload.js", // ersetzen mit jQuery ajax()
                "ajaxupload.min.js",
                "jquery.cookie.js", // noch ok
                // "jquery.cookie.min.js",
                "jquery.epiclock.js", // ersetzen mit jQueryUI progressbar() - bzw ein eigenes mit Timer
                "jquery.epiclock.min.js",
                "jquery.imagefit.js", // letzte update 2015 - ersetzen mit jQueryUI dialog() (oder komplett raus) oder
                "jquery.password-checker.js", // noch ok - hier schreiben wir ein eigenes
                // "jquery.password-checker.min.js",
                "jquery.prettyPhoto.js",
                "jquery.prettyPhoto.min.js",
                "jquery.raty.js", // hier schreiben wir ein eigenes
                "jquery.raty.min.js",
                "jquery.sticky.js", // nicht in benutzung
            ],
        ];
        $_jQueryUI = [
            "files" => [
                "ui/jquery-ui-pmx-core.min.js", // core - v1.10.2
                "ui/jquery.ui.core.min.js", // core - v1.10.3
                // All diese Dateien sind in jQueryUI 1.13.1 mit enthalten
                // "ui/jquery.ui.tabs.min.js",
                // "ui/jquery.ui.tooltip.min.js",
                // "ui/jquery.ui.accordion.min.js",
                // "ui/jquery.ui.autocomplete.min.js",
                // "ui/jquery.ui.button.min.js",
                // "ui/jquery.ui.datepicker.min.js",
                // "ui/jquery.ui.dialog.min.js",
                // "ui/jquery.ui.draggable.min.js",
                // "ui/jquery.ui.droppable.min.js",
                // "ui/jquery.ui.effect.min.js",
                // "ui/jquery.ui.effect-blind.min.js",
                // "ui/jquery.ui.effect-bounce.min.js",
                // "ui/jquery.ui.effect-clip.min.js",
                // "ui/jquery.ui.effect-drop.min.js",
                // "ui/jquery.ui.effect-explode.min.js",
                // "ui/jquery.ui.effect-fade.min.js",
                // "ui/jquery.ui.effect-fold.min.js",
                // "ui/jquery.ui.effect-highlight.min.js",
                // "ui/jquery.ui.effect-pulsate.min.js",
                // "ui/jquery.ui.effect-scale.min.js",
                // "ui/jquery.ui.effect-shake.min.js",
                // "ui/jquery.ui.effect-slide.min.js",
                // "ui/jquery.ui.effect-transfer.min.js",
                // "ui/jquery.ui.menu.min.js",
                // "ui/jquery.ui.progressbar.min.js",
                // "ui/jquery.ui.resizable.min.js",
                // "ui/jquery.ui.selectable.min.js",
                // "ui/jquery.ui.slider.min.js",
                // "ui/jquery.ui.sortable.min.js",
                // "ui/jquery.ui.spinner.min.js",
            ],
            "plugins" => [
                "ui/jquery.ui.tabs.noui.js", // ersetzen mit jQueryUI tabs()
                "ui/jquery.ui.tabs.noui.min.js",
            ],
        ];

        if (!self::_isFileHttp($file)) {
            // remove path if exists
            $file = str_replace(self::$__jquery['path'], "", $file);
        
            if (self::$__jQueryOldErrorMessage === true) {
                if (in_array($file, $_jQuery["files"]) || in_array($file, $_jQuery["plugins"])) {
                    trigger_error($errorMessage . " - Ready for upgrade jQuery - Please not use - " . $file, E_USER_DEPRECATED);
                }
                if (in_array($file, $_jQueryUI["files"]) || in_array($file, $_jQueryUI["plugins"])) {
                    trigger_error($errorMessage . " - Ready for upgrade jQueryUI - Please not use - " . $file, E_USER_DEPRECATED);
                }
            }
        }
    }

    // TODO: extra function um die Version zu lesen - und nicht von ausserhalb so wie bei Admin/Dashboard

    /**
     * pmxHeader::__callStatic()
     * einfach nur dummy um fehlende methoden zu simulieren...
     *
     * @param string $name
     * @param array $arguments
     * @return bol false
     */
    // public static function __callStatic ($name, $arguments = array() ){
    // trigger_error('Call to undefined method ' .__CLASS__. '::' . $name . '()', E_USER_WARNING);
    // return false;
    // }
}

?>
